<?php
// Webhook Keeper - Auto-reconnect webhook if disconnected
// This file should be called via cron job every 5-10 minutes
// Example cron: */5 * * * * curl -s https://yourdomain.com/webhook_keeper.php > /dev/null 2>&1

require_once 'config.php';

// Get current webhook URL
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
$domain = $protocol . '://' . $_SERVER['HTTP_HOST'];
$script_path = dirname($_SERVER['REQUEST_URI'] ?? '/');
if ($script_path === '/' || $script_path === '\\') {
    $webhook_url = $domain . '/basic_bot.php';
} else {
    $webhook_url = rtrim($domain . $script_path, '/') . '/basic_bot.php';
}

// Get webhook info from Telegram
function getWebhookInfo() {
    global $BOT_TOKEN;
    $api_url = "https://api.telegram.org/bot$BOT_TOKEN/getWebhookInfo";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code !== 200 || !$response) {
        return ['ok' => false, 'error' => 'Failed to connect to Telegram API'];
    }
    
    return json_decode($response, true);
}

// Set webhook
function setWebhook($url) {
    global $BOT_TOKEN;
    $api_url = "https://api.telegram.org/bot$BOT_TOKEN/setWebhook";
    
    $data = [
        'url' => $url,
        'drop_pending_updates' => false, // Keep pending updates
        'allowed_updates' => ['message', 'callback_query'] // Only what we need
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code !== 200 || !$response) {
        return ['ok' => false, 'error' => 'Failed to connect to Telegram API'];
    }
    
    return json_decode($response, true);
}

// Verify webhook URL is accessible
function verifyWebhookURL($url) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['test' => true]));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 3);
    curl_setopt($ch, CURLOPT_NOBODY, false);
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return $http_code === 200;
}

// Log webhook events
function logWebhookEvent($message) {
    $log_file = 'webhook_keeper.log';
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[$timestamp] $message\n";
    file_put_contents($log_file, $log_entry, FILE_APPEND | LOCK_EX);
}

// Main webhook keeper logic
$webhook_info = getWebhookInfo();

if (!$webhook_info['ok']) {
    logWebhookEvent("ERROR: Failed to get webhook info - " . ($webhook_info['error'] ?? 'Unknown error'));
    exit(1);
}

$current_webhook = $webhook_info['result'];
$needs_reconnect = false;
$reason = '';

// Check if webhook is set
if (empty($current_webhook['url'])) {
    $needs_reconnect = true;
    $reason = 'Webhook is not set';
} 
// Check if webhook URL matches
elseif ($current_webhook['url'] !== $webhook_url) {
    $needs_reconnect = true;
    $reason = 'Webhook URL mismatch: ' . $current_webhook['url'] . ' vs ' . $webhook_url;
}
// Check if there are too many pending updates (indicates webhook might be broken)
elseif (isset($current_webhook['pending_update_count']) && $current_webhook['pending_update_count'] > 100) {
    $needs_reconnect = true;
    $reason = 'Too many pending updates: ' . $current_webhook['pending_update_count'];
}
// Check if there's a recent error
elseif (isset($current_webhook['last_error_date']) && $current_webhook['last_error_date']) {
    $error_age = time() - $current_webhook['last_error_date'];
    // If error is less than 1 hour old, reconnect
    if ($error_age < 3600) {
        $needs_reconnect = true;
        $reason = 'Recent webhook error: ' . ($current_webhook['last_error_message'] ?? 'Unknown error');
    }
}

// Verify webhook URL is accessible
if (!$needs_reconnect) {
    if (!verifyWebhookURL($webhook_url)) {
        $needs_reconnect = true;
        $reason = 'Webhook URL is not accessible';
    }
}

// Reconnect if needed
if ($needs_reconnect) {
    logWebhookEvent("WARNING: Webhook needs reconnection - $reason");
    
    // Verify URL is accessible before setting
    if (!verifyWebhookURL($webhook_url)) {
        logWebhookEvent("ERROR: Cannot reconnect - webhook URL is not accessible: $webhook_url");
        exit(1);
    }
    
    // Set webhook
    $result = setWebhook($webhook_url);
    
    if ($result['ok']) {
        logWebhookEvent("SUCCESS: Webhook reconnected successfully to: $webhook_url");
        
        // If called from browser, show message
        if (php_sapi_name() !== 'cli' && !isset($_GET['silent'])) {
            header('Content-Type: text/plain');
            echo "✅ Webhook reconnected successfully!\n";
            echo "URL: $webhook_url\n";
            echo "Reason: $reason\n";
        }
    } else {
        logWebhookEvent("ERROR: Failed to reconnect webhook - " . ($result['description'] ?? 'Unknown error'));
        
        if (php_sapi_name() !== 'cli' && !isset($_GET['silent'])) {
            header('Content-Type: text/plain');
            echo "❌ Failed to reconnect webhook\n";
            echo "Error: " . ($result['description'] ?? 'Unknown error') . "\n";
        }
        exit(1);
    }
} else {
    logWebhookEvent("OK: Webhook is active and healthy - URL: " . $current_webhook['url']);
    
    if (php_sapi_name() !== 'cli' && !isset($_GET['silent'])) {
        header('Content-Type: text/plain');
        echo "✅ Webhook is active and healthy\n";
        echo "URL: " . $current_webhook['url'] . "\n";
        echo "Pending Updates: " . ($current_webhook['pending_update_count'] ?? 0) . "\n";
    }
}

exit(0);
?>

